<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreDestinationRequest;
use App\Http\Requests\UpdateDestinationRequest;
use App\Models\Destination;
use Illuminate\Support\Str;

class DestinationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $destinations = Destination::withCount(['places'])->orderBy('name')->get();

        return view('admin.destination.index', compact('destinations'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $destination = new Destination;

        return view('admin.destination.create', compact('destination'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreDestinationRequest $request)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['thumbnail'] = $request->file('thumbnail')->store('destinations', 'public');
        $data['slug'] = Str::slug($data['name']);

        try {
            $destination = Destination::updateOrCreate([
                'slug' => $data['slug'],
                'country_id' => $data['country_id'],
                'state_id' => $data['state_id'],
            ], $data);
            if ($request->has('places')) {
                foreach ($request->places as $place) {
                    $destination->places()->updateOrCreate([
                        'slug' => Str::slug($place),
                    ], [
                        'name' => $place,
                    ]);
                }
            }

            return redirect()->route('admin.destination.index')->with('success', 'Destination created successfully.');
        } catch (\Exception $e) {
            return $e->getMessage();

            return redirect()->back()->with('error', 'Destination could not be created.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Destination $destination)
    {
        return view('admin.destination.show', compact('destination'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Destination $destination)
    {
        return view('admin.destination.create', compact('destination'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateDestinationRequest $request, Destination $destination)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['name']);
        if ($request->has('thumbnail')) {
            $data['thumbnail'] = $request->file('thumbnail')->store('destinations', 'public');
        }

        try {
            $destination->update($data);
            if ($request->has('places')) {
                $destination->places()->delete();
                foreach ($request->places as $place) {
                    $destination->places()->updateOrCreate([
                        'slug' => Str::slug($place),
                    ], [
                        'name' => $place,
                    ]);
                }
            }

            return redirect()->route('admin.destination.index')->with('success', 'Destination updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Destination could not be updated.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Destination $destination)
    {
        try {
            $path = public_path('storage/'.$destination->thumbnail);
            $destination->delete();
            $destination->places()->delete();
            if (file_exists($path)) {
                @unlink($path);
            }

            return response()->json(['message' => 'Destination deleted successfully.', 'status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Destination could not be deleted.', 'status' => 'error'], 500);
        }
    }
}
